<?php
/**
 * MLPC License System - Ana API
 */

require_once __DIR__ . '/config.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    header('Access-Control-Allow-Origin: ' . ALLOWED_ORIGINS);
    header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization, X-API-Secret, X-Admin-Key');
    http_response_code(204);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Sadece POST metodu desteklenir', 405);
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$input = json_decode(file_get_contents('php://input'), true) ?? [];

switch ($action) {
    case 'check': checkLicense($input); break;
    case 'list': requireAdmin(); listLicenses($input); break;
    case 'create': requireAdmin(); createLicense($input); break;
    case 'update': requireAdmin(); updateLicense($input); break;
    case 'delete': requireAdmin(); deleteLicense($input); break;
    case 'logs': requireAdmin(); getLogs($input); break;
    case 'test': testConnection(); break;
    default: sendError('Geçersiz action', 400);
}

function requireAdmin() {
    if (($_SERVER['HTTP_X_ADMIN_KEY'] ?? '') !== ADMIN_KEY) {
        sendError('Yetkisiz erişim', 401);
    }
}

function testConnection() {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT COUNT(*) as count FROM licenses");
        sendSuccess(['license_count' => $stmt->fetch()['count']], 'Bağlantı başarılı');
    } catch (Exception $e) {
        sendError('Veritabanı hatası: ' . $e->getMessage(), 500);
    }
}

function checkLicense($input) {
    if (($_SERVER['HTTP_X_API_SECRET'] ?? '') !== API_SECRET) {
        sendError('Yetkisiz erişim', 401);
    }
    
    $licenseKey = trim($input['license_key'] ?? '');
    $domain = normalizeDomain($input['domain'] ?? '');
    
    if (empty($licenseKey) || empty($domain)) {
        sendError('Lisans anahtarı ve domain gerekli');
    }
    
    $db = getDB();
    $ip = getClientIP();
    $ua = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    $logCheck = function($licenseId, $result, $reason = null) use ($db, $licenseKey, $domain, $ip, $ua) {
        try {
            $stmt = $db->prepare("INSERT INTO license_logs (id, license_id, license_key, domain, ip_address, user_agent, result, failure_reason, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
            $stmt->execute([generateUUID(), $licenseId, $licenseKey, $domain, $ip, $ua, $result, $reason]);
        } catch (Exception $e) {}
    };
    
    $stmt = $db->prepare("SELECT * FROM licenses WHERE license_key = ? LIMIT 1");
    $stmt->execute([$licenseKey]);
    $license = $stmt->fetch();
    
    if (!$license) {
        $logCheck(null, 'failed', 'Lisans bulunamadı');
        sendJSON(['valid' => false, 'message' => 'Geçersiz lisans anahtarı']);
    }
    
    if (normalizeDomain($license['domain']) !== $domain) {
        $logCheck($license['id'], 'failed', 'Domain uyuşmazlığı');
        sendJSON(['valid' => false, 'message' => 'Bu lisans başka bir domain için kayıtlı']);
    }
    
    if ($license['status'] === 'banned') {
        $logCheck($license['id'], 'failed', 'Yasaklı');
        sendJSON(['valid' => false, 'message' => 'Lisans iptal edilmiş']);
    }
    
    if ($license['status'] === 'expired') {
        $logCheck($license['id'], 'failed', 'Süresi dolmuş');
        sendJSON(['valid' => false, 'message' => 'Lisans süresi dolmuş']);
    }
    
    if (!empty($license['expire_date'])) {
        $expireDate = new DateTime($license['expire_date']);
        $expireDate->setTime(23, 59, 59);
        if (new DateTime() > $expireDate) {
            $db->prepare("UPDATE licenses SET status = 'expired', updated_at = NOW() WHERE id = ?")->execute([$license['id']]);
            $logCheck($license['id'], 'failed', 'Tarih geçmiş');
            sendJSON(['valid' => false, 'message' => 'Lisans süresi dolmuş']);
        }
    }
    
    $logCheck($license['id'], 'success');
    sendJSON(['valid' => true, 'status' => $license['status'], 'message' => 'Lisans geçerli', 'expires_at' => $license['expire_date']]);
}

function listLicenses($input) {
    $db = getDB();
    $limit = min(intval($input['limit'] ?? 100), 500);
    $stmt = $db->prepare("SELECT * FROM licenses ORDER BY created_at DESC LIMIT ?");
    $stmt->execute([$limit]);
    sendSuccess(['data' => $stmt->fetchAll(), 'count' => $stmt->rowCount()]);
}

function createLicense($input) {
    $domain = normalizeDomain($input['domain'] ?? '');
    $licenseKey = trim($input['license_key'] ?? '');
    $status = in_array($input['status'] ?? '', ['active', 'expired', 'banned']) ? $input['status'] : 'active';
    
    if (empty($domain) || empty($licenseKey)) {
        sendError('Domain ve lisans anahtarı gerekli');
    }
    
    $db = getDB();
    $check = $db->prepare("SELECT id FROM licenses WHERE license_key = ?");
    $check->execute([$licenseKey]);
    if ($check->fetch()) sendError('Bu lisans anahtarı zaten mevcut');
    
    $id = generateUUID();
    $stmt = $db->prepare("INSERT INTO licenses (id, domain, license_key, status, expire_date, owner_email, owner_name, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())");
    $stmt->execute([$id, $domain, $licenseKey, $status, $input['expire_date'] ?? null, $input['owner_email'] ?? null, $input['owner_name'] ?? null]);
    sendSuccess(['id' => $id], 'Lisans oluşturuldu');
}

function updateLicense($input) {
    $id = $input['id'] ?? '';
    if (empty($id)) sendError('Lisans ID gerekli');
    
    $db = getDB();
    $updates = ['updated_at = NOW()'];
    $params = [];
    
    foreach (['domain', 'license_key', 'status', 'expire_date', 'owner_email', 'owner_name'] as $field) {
        if (isset($input[$field])) {
            $val = $field === 'domain' ? normalizeDomain($input[$field]) : ($input[$field] ?: null);
            if ($field === 'status' && !in_array($val, ['active', 'expired', 'banned'])) continue;
            $updates[] = "$field = ?";
            $params[] = $val;
        }
    }
    
    $params[] = $id;
    $db->prepare("UPDATE licenses SET " . implode(', ', $updates) . " WHERE id = ?")->execute($params);
    sendSuccess([], 'Lisans güncellendi');
}

function deleteLicense($input) {
    $id = $input['id'] ?? '';
    if (empty($id)) sendError('Lisans ID gerekli');
    
    $db = getDB();
    $stmt = $db->prepare("DELETE FROM licenses WHERE id = ?");
    $stmt->execute([$id]);
    sendSuccess([], $stmt->rowCount() ? 'Lisans silindi' : 'Lisans bulunamadı');
}

function getLogs($input) {
    $db = getDB();
    $limit = min(intval($input['limit'] ?? 100), 500);
    $stmt = $db->prepare("SELECT * FROM license_logs ORDER BY created_at DESC LIMIT ?");
    $stmt->execute([$limit]);
    sendSuccess(['data' => $stmt->fetchAll(), 'count' => $stmt->rowCount()]);
}
